-- Script untuk menginisialisasi stok produk berdasarkan bahan baku yang tersedia
-- Jalankan script ini setelah menjalankan add_stok_field.sql

USE wargas;

-- Pastikan field stok sudah ada di tabel produk
ALTER TABLE produk ADD COLUMN IF NOT EXISTS stok INT(11) DEFAULT 0 AFTER harga;

-- Update stok produk berdasarkan bahan baku yang tersedia
-- Menggunakan logika BOM (Bill of Materials)

-- Fungsi untuk menghitung stok maksimal produk berdasarkan bahan baku tersedia
DELIMITER $$
CREATE FUNCTION IF NOT EXISTS CalculateProductStock(product_code VARCHAR(100)) 
RETURNS INT
READS SQL DATA
DETERMINISTIC
BEGIN
    DECLARE min_stock INT DEFAULT 2147483647; -- PHP_INT_MAX equivalent
    DECLARE max_produk_dari_bahan INT;
    DECLARE kebutuhan DECIMAL(10,2);
    DECLARE stok_bahan DECIMAL(10,2);
    DECLARE done INT DEFAULT FALSE;
    
    DECLARE bom_cursor CURSOR FOR 
        SELECT bp.kebutuhan, COALESCE(i.qty, 0) as qty
        FROM bom_produk bp 
        LEFT JOIN inventory i ON bp.kode_bk = i.kode_bk 
        WHERE bp.kode_produk = product_code;
    
    DECLARE CONTINUE HANDLER FOR NOT FOUND SET done = TRUE;
    
    OPEN bom_cursor;
    
    read_loop: LOOP
        FETCH bom_cursor INTO kebutuhan, stok_bahan;
        IF done THEN
            LEAVE read_loop;
        END IF;
        
        IF kebutuhan > 0 THEN
            SET max_produk_dari_bahan = FLOOR(stok_bahan / kebutuhan);
            IF max_produk_dari_bahan < min_stock THEN
                SET min_stock = max_produk_dari_bahan;
            END IF;
        END IF;
    END LOOP;
    
    CLOSE bom_cursor;
    
    -- Jika tidak ada bahan baku atau semua bahan baku habis
    IF min_stock = 2147483647 THEN
        SET min_stock = 0;
    END IF;
    
    -- Pastikan stok tidak negatif
    RETURN GREATEST(0, min_stock);
END$$
DELIMITER ;

-- Update stok untuk semua produk
UPDATE produk SET stok = CalculateProductStock(kode_produk);

-- Tampilkan hasil
SELECT 
    p.kode_produk,
    p.nama as nama_produk,
    p.stok as stok_produk,
    GROUP_CONCAT(CONCAT(bp.kode_bk, ':', bp.kebutuhan, ' (', COALESCE(i.qty, 0), ')') SEPARATOR ', ') as bahan_baku_dibutuhkan
FROM produk p
LEFT JOIN bom_produk bp ON p.kode_produk = bp.kode_produk
LEFT JOIN inventory i ON bp.kode_bk = i.kode_bk
GROUP BY p.kode_produk, p.nama, p.stok
ORDER BY p.kode_produk;

-- Hapus fungsi setelah selesai (opsional)
-- DROP FUNCTION IF EXISTS CalculateProductStock; 